---
title: DDoS managed rulesets configuration using Terraform · Cloudflare Terraform docs
description: "This page provides examples of configuring DDoS managed rulesets
  in your zone or account using Terraform. It covers the following
  configurations:"
lastUpdated: 2025-08-20T20:59:04.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/terraform/additional-configurations/ddos-managed-rulesets/
  md: https://developers.cloudflare.com/terraform/additional-configurations/ddos-managed-rulesets/index.md
---

This page provides examples of configuring [DDoS managed rulesets](https://developers.cloudflare.com/ddos-protection/managed-rulesets/) in your zone or account using Terraform. It covers the following configurations:

* [Example: Configure HTTP DDoS Attack Protection](#example-configure-http-ddos-attack-protection)
* [Example: Configure Network-layer DDoS Attack Protection](#example-configure-network-layer-ddos-attack-protection)
* [Use case: Mitigate large HTTP DDoS attacks and monitor flagged traffic](#use-case-mitigate-large-http-ddos-attacks-and-monitor-flagged-traffic)

DDoS managed rulesets are always enabled. Depending on your Cloudflare services, you may be able to adjust their behavior.

If you are using the Cloudflare API, refer to the following resources:

* [Configure HTTP DDoS Attack Protection via API](https://developers.cloudflare.com/ddos-protection/managed-rulesets/http/http-overrides/configure-api/)
* [Configure Network-layer DDoS Attack Protection via API](https://developers.cloudflare.com/ddos-protection/managed-rulesets/network/network-overrides/configure-api/)

For more information on deploying and configuring rulesets using the Rulesets API, refer to [Work with managed rulesets](https://developers.cloudflare.com/ruleset-engine/managed-rulesets/) in the Ruleset Engine documentation.

## Before you start

### Obtain the necessary account, zone, and managed ruleset IDs

The Terraform configurations provided in this page need the zone ID (or account ID) of the zone/account where you will deploy the managed rulesets.

* To retrieve the list of accounts you have access to, including their IDs, use the [List accounts](https://developers.cloudflare.com/api/resources/accounts/methods/list/) operation.
* To retrieve the list of zones you have access to, including their IDs, use the [List zones](https://developers.cloudflare.com/api/resources/zones/methods/list/) operation.

The deployment of managed rulesets via Terraform requires that you use the ruleset IDs. To find the IDs of managed rulesets, use the [List account rulesets](https://developers.cloudflare.com/api/resources/rulesets/methods/list/) operation. The response will include the description and IDs of existing managed rulesets.

### (Optional) Delete existing rulesets to start from scratch

Terraform assumes that it has complete control over account and zone rulesets. If you already have rulesets configured in your account or zone, do one of the following:

* [Import existing rulesets to Terraform](https://developers.cloudflare.com/terraform/advanced-topics/import-cloudflare-resources/) using the `cf-terraforming` tool. Recent versions of the tool can generate resource definitions for existing rulesets and import their configuration to Terraform state.
* Start from scratch by [deleting existing rulesets](https://developers.cloudflare.com/ruleset-engine/rulesets-api/delete/#delete-ruleset) (account and zone rulesets with `"kind": "root"` and `"kind": "zone"`, respectively) and then defining your rulesets configuration in Terraform.

***

## Example: Configure HTTP DDoS Attack Protection

This example configures the [HTTP DDoS Attack Protection](https://developers.cloudflare.com/ddos-protection/managed-rulesets/http/) managed ruleset for a zone using Terraform.

Note

Terraform code snippets below refer to the v4 SDK only.

```tf
resource "cloudflare_ruleset" "zone_level_http_ddos_config" {
  zone_id     = "<ZONE_ID>"
  name        = "HTTP DDoS Attack Protection entry point ruleset"
  description = ""
  kind        = "zone"
  phase       = "ddos_l7"


  rules {
    action = "execute"
    action_parameters {
      # Cloudflare L7 DDoS Attack Protection Ruleset
      id = "4d21379b4f9f4bb088e0729962c8b3cf"
      overrides {
        action = "block"
        sensitivity_level = "default"
        rules {
          # Adaptive DDoS Protection based on Locations (Available only to Enterprise zones with Advanced DDoS service)
          id = "a8c6333711ff4b0a81371d1c444be2c3"
          sensitivity_level = "default"
          action = "managed_challenge"
        }
        rules {
          # Adaptive DDoS Protection based on User-Agents (Available only to Enterprise zones with Advanced DDoS service)
          id = "7709d496081e458899c1e3a6e4fe8e55"
          sensitivity_level = "default"
          action = "managed_challenge"
        }
        rules {
          # HTTP requests causing a high number of origin errors.
          id = "dd42da7baabe4e518eaf11c393596a9d"
          sensitivity_level = "default"
          action = "managed_challenge"
        }
      }
    }
    expression = "true"
    description = "Zone-wide HTTP DDoS Override"
    enabled = true
  }
}
```

For more information about HTTP DDoS Attack Protection, refer to [HTTP DDoS Attack Protection managed ruleset](https://developers.cloudflare.com/ddos-protection/managed-rulesets/http/).

## Example: Configure Network-layer DDoS Attack Protection

This example configures the [Network-layer DDoS Attack Protection](https://developers.cloudflare.com/ddos-protection/managed-rulesets/network/) managed ruleset for an account using Terraform, changing the sensitivity level of rule with ID 599dab0942ff4898ac1b7797e954e98b to `low` using an override.

Important

* Only Magic Transit and Spectrum customers on an Enterprise plan can configure this managed ruleset using overrides.
* This managed ruleset only supports overrides at the account level.

Note

Terraform code snippets below refer to the v4 SDK only.

```tf
resource "cloudflare_ruleset" "account_level_network_ddos_config" {
  account_id  = "<ACCOUNT_ID>"
  name        = "Network-layer DDoS Attack Protection entry point ruleset"
  description = ""
  kind        = "root"
  phase       = "ddos_l4"


  rules {
    ref         = "override_l7_ddos_ruleset_dst_ip"
    description = "Override the HTTP DDoS Attack Protection managed ruleset"
    expression  = "ip.dst in { 192.0.2.0/24 }"
    action      = "execute"
    action_parameters {
      # Cloudflare L3/4 DDoS Attack Protection Ruleset
      id = "3b64149bfa6e4220bbbc2bd6db589552"
      overrides {
        rules {
          # Rule: Generic high-volume UDP traffic flows.
          id                = "599dab0942ff4898ac1b7797e954e98b"
          sensitivity_level = "low"
        }
      }
    }
  }
}
```

For more information about Network-layer DDoS Attack Protection, refer to [Network-layer DDoS Attack Protection managed ruleset](https://developers.cloudflare.com/ddos-protection/managed-rulesets/network/).

***

## Use case: Mitigate large HTTP DDoS attacks and monitor flagged traffic

In the following example, a customer is concerned about false positives, but wants to get protection against large HTTP DDoS attacks. The two rules, containing two overrides each, in their [HTTP DDoS protection](https://developers.cloudflare.com/ddos-protection/managed-rulesets/http/) configuration will have the following behavior:

1. Mitigate any large HTTP DDoS attacks by configuring a rule with a *Low* [sensitivity level](https://developers.cloudflare.com/ddos-protection/managed-rulesets/http/override-parameters/#sensitivity-level) and a *Block* action.
2. Monitor traffic being flagged by the DDoS protection system by configuring a rule with the default sensitivity level (*High*) and a *Log* action.

The order of the rules is important: the rule with the highest sensitivity level must come after the rule with the lowest sensitivity level, otherwise it will never be evaluated.

Important considerations

* When a DDoS attack mitigation is ongoing, Cloudflare will check the rules order and apply the first one that matches both the expression and the sensitivity level.
* Since rules are evaluated in order and the first one to match the conditions of both the expression and the sensitivity level will get applied, take care when editing and reordering existing rules. Changing a rule from Block to Log may allow attack traffic to reach your web property.
* Overrides will not affect read-only rules in the managed ruleset.

Note

Terraform code snippets below refer to the v4 SDK only.

```tf
variable "zone_id" {
  default = "<ZONE_ID>"
}


resource "cloudflare_ruleset" "zone_level_http_ddos_config" {
  zone_id     = var.zone_id
  name        = "HTTP DDoS - Terraform managed"
  description = ""
  kind        = "zone"
  phase       = "ddos_l7"


  # The resource configuration contains two rules:
  #  1. The first rule has the lowest sensitivity level (highest threshold)
  #     and it will block attacks.
  #  2. The second rule has a higher sensitivity level (lower threshold) and
  #     will only apply a Log action.
  #
  # In practice, evaluation stops whenever a rule matches both the expression
  # and the threshold, so the rule order is important:
  #   - When the traffic rate is below the (low) threshold of the default
  #     sensitivity level ('High'), no rules match (no action is applied).
  #   - When the traffic rate is between the thresholds of the 'Low' and
  #     default ('High') sensitivity levels, the first rule does not match,
  #     but the second rule does (traffic gets logged).
  #   - When the traffic rate goes above the (high) threshold of the 'Low'
  #     sensitivity level, the first rule matches (traffic gets blocked).
  #
  # The DDoS protection systems will still apply mitigation actions to incoming
  # traffic when rates exceed the threshold of the _Essentially Off_ sensitivity
  # level.


  rules {
    ref         = "l7_ddos_block_traffic_low_threshold"
    description = "At the low sensitivity threshold, block the traffic"
    expression  = "true"
    action      = "execute"
    action_parameters {
      # Cloudflare L7 DDoS Attack Protection Ruleset
      id = "4d21379b4f9f4bb088e0729962c8b3cf"
      overrides {
        rules {
          # Rule: HTTP requests from known botnet (signature #4).
          id                = "29d170ba2f004cc787b1ac272c9e04e7"
          sensitivity_level = "low"
          action            = "block"
        }
        rules {
          # Rule: HTTP requests with unusual HTTP headers or URI path (signature #16).
          id                = "60a48054bbcf4014ac63c44f1712a123"
          sensitivity_level = "low"
          action            = "block"
        }
      }
    }
  }


  rules {
    ref         = "l7_ddos_log_default_threshold"
    description = "At the default sensitivity threshold, log to see if any legitimate traffic gets caught"
    expression  = "true"
    action      = "execute"
    action_parameters {
      # Cloudflare L7 DDoS Attack Protection Ruleset
      id = "4d21379b4f9f4bb088e0729962c8b3cf"
      overrides {
        rules {
          # Rule: HTTP requests from known botnet (signature #4).
          id                = "29d170ba2f004cc787b1ac272c9e04e7"
          sensitivity_level = "default"
          action            = "log"
        }
        rules {
          # Rule: HTTP requests with unusual HTTP headers or URI path (signature #16).
          id                = "60a48054bbcf4014ac63c44f1712a123"
          sensitivity_level = "default"
          action            = "log"
        }
      }
    }
  }
}
```
